// Scene Manager
const scenes = [];

function createScene(containerId, initFn, animateFn) {
    const container = document.getElementById(containerId);
    if (!container) return;

    const scene = new THREE.Scene();
    const camera = new THREE.PerspectiveCamera(75, container.clientWidth / container.clientHeight, 0.1, 1000);
    const renderer = new THREE.WebGLRenderer({ alpha: true, antialias: true });

    renderer.setSize(container.clientWidth, container.clientHeight);
    container.appendChild(renderer.domElement);

    // Resize handler
    window.addEventListener('resize', () => {
        camera.aspect = container.clientWidth / container.clientHeight;
        camera.updateProjectionMatrix();
        renderer.setSize(container.clientWidth, container.clientHeight);
    });

    const context = { scene, camera, renderer };
    initFn(context);

    scenes.push({ context, animateFn });
}

// Global Animation Loop
function animate() {
    requestAnimationFrame(animate);
    scenes.forEach(({ context, animateFn }) => {
        animateFn(context);
        context.renderer.render(context.scene, context.camera);
    });
}

// --- 1. HERO SCENE ---
createScene('hero-canvas', ({ scene, camera }) => {
    camera.position.z = 5;

    // Lights with massive intensity
    const ambientLight = new THREE.AmbientLight(0xffffff, 0.5);
    scene.add(ambientLight);

    const spotLight = new THREE.SpotLight(0x00f3ff, 1);
    spotLight.position.set(10, 10, 10);
    scene.add(spotLight);

    const pointLight = new THREE.PointLight(0xbd00ff, 1); // Purple
    pointLight.position.set(-10, -10, 10);
    scene.add(pointLight);

    // Main Octahedron (Wireframe for simplified cool look)
    const geometry = new THREE.OctahedronGeometry(1.5, 0);
    const material = new THREE.MeshBasicMaterial({
        color: 0x00f3ff, // Cyan
        wireframe: true
    });
    scene.userData.octahedron = new THREE.Mesh(geometry, material);
    scene.add(scene.userData.octahedron);

    // Outer Icosahedron (Ghost shell)
    const geo2 = new THREE.IcosahedronGeometry(2.5, 1);
    const mat2 = new THREE.MeshBasicMaterial({
        color: 0xffffff,
        wireframe: true,
        transparent: true,
        opacity: 0.1
    });
    scene.userData.outerShell = new THREE.Mesh(geo2, mat2);
    scene.add(scene.userData.outerShell);

    // Torus Rings
    const torusGeo = new THREE.TorusGeometry(3, 0.02, 16, 100);
    const torusMat = new THREE.MeshBasicMaterial({ color: 0xccff00 }); // Lime
    scene.userData.torus = new THREE.Mesh(torusGeo, torusMat);
    scene.userData.torus.rotation.x = Math.PI / 2;
    scene.add(scene.userData.torus);

}, ({ scene }) => {
    const time = Date.now() * 0.001;

    if (scene.userData.octahedron) {
        scene.userData.octahedron.rotation.x = time * 0.5;
        scene.userData.octahedron.rotation.y = time * 0.2;
        // Pulse scale
        const scale = 1 + Math.sin(time * 2) * 0.1;
        scene.userData.octahedron.scale.set(scale, scale, scale);
    }

    if (scene.userData.outerShell) {
        scene.userData.outerShell.rotation.y = -time * 0.1;
    }

    if (scene.userData.torus) {
        scene.userData.torus.rotation.x = (Math.PI / 2) + Math.sin(time) * 0.2;
        scene.userData.torus.rotation.y = time * 0.1;
    }
});

// --- 2. TERMINAL SCENE ---
createScene('terminal-container', ({ scene, camera }) => {
    camera.position.z = 5;

    // Box Geometry
    const geometry = new THREE.BoxGeometry(3, 2, 0.2);
    const material = new THREE.MeshNormalMaterial({ wireframe: true }); // Placeholder simple vibe
    const cube = new THREE.Mesh(geometry, material);
    scene.userData.terminal = cube;
    scene.add(cube);

    // Add text plane? (Simplified for vanilla Three.js without FontLoader)
    // Maybe just a green plane inside
    const planeGeo = new THREE.PlaneGeometry(2.8, 1.8);
    const planeMat = new THREE.MeshBasicMaterial({
        color: 0x00ff00,
        transparent: true,
        opacity: 0.1,
        side: THREE.DoubleSide
    });
    const screen = new THREE.Mesh(planeGeo, planeMat);
    screen.position.z = 0.11;
    cube.add(screen);

}, ({ scene }) => {
    const time = Date.now() * 0.001;

    if (scene.userData.terminal) {
        scene.userData.terminal.rotation.x = Math.sin(time * 0.5) * 0.1;
        scene.userData.terminal.rotation.y = Math.sin(time * 0.3) * 0.1;
    }
});

// --- 3. WAVEFORM SCENE ---
createScene('waveform-container', ({ scene, camera }) => {
    camera.position.z = 6;
    camera.position.y = 2;
    camera.lookAt(0, 0, 0);

    scene.userData.bars = [];

    // Create cylinder bars
    const count = 20;
    const radius = 3;

    for (let i = 0; i < count; i++) {
        const angle = (i / count) * Math.PI * 2;
        const geometry = new THREE.CylinderGeometry(0.1, 0.1, 1, 8);
        const material = new THREE.MeshBasicMaterial({ color: 0xbd00ff });
        const bar = new THREE.Mesh(geometry, material);

        bar.position.set(Math.cos(angle) * radius, 0, Math.sin(angle) * radius);
        scene.add(bar);
        scene.userData.bars.push({ mesh: bar, angle: angle });
    }

}, ({ scene }) => {
    const time = Date.now() * 0.003;

    scene.userData.bars.forEach((barData, i) => {
        const height = 1 + Math.sin(time + i) * 2; // Wave effect
        barData.mesh.scale.y = Math.max(0.1, height);
        barData.mesh.position.y = barData.mesh.scale.y / 2; // Keep base at 0

        // Color cycle
        const hue = (time * 0.1 + i * 0.05) % 1;
        barData.mesh.material.color.setHSL(hue, 1, 0.5);
    });

    scene.rotation.y = time * 0.1;
});

// Start Animation
animate();
